<?php

namespace App\Http\Controllers;

use App\Models\Galeria;
use Illuminate\Http\Request;
use App\Models\Projecto;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
class ProjectosController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {

        $search = $request->input('search');

        $projectos = Projecto::query()->when($search, function ($query, $search) {
            $query->Where('nome_projecto', '=', "%{$search}%")
            ->orWhere('tipo_servico', '=', "%{$search}%")
            ->orWhere('referencia_projecto', '=', "%{$search}%")
            ->orWhere('tipo_projecto', '=', "%{$search}%");
        })->paginate(10);
        return view('dashboard.admin.pages.projectos.projectos', ['projectos' => $projectos]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('dashboard.admin.pages.projectos.add-projectos');

    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {

        $validator = Validator::make(
            $request->all(),
            [
                'nome_projecto'       => 'required|string|max:50',
                'endereco'            => 'required|string|max:200',
                'nome_cliente'        => 'required|string|max:50',
                'engenheiro_resp'     => 'required|string|max:50',

                'orcamento_prev'      => 'required|numeric|min:0',
                'custo_total'         => 'nullable|numeric|min:0',

                'moeda'               => 'required|string|max:20',
                'estado'              => 'required|string|max:50',
                'tipo_servico'        => 'required|string|max:100',
                'tipo_projecto'       => 'required|string|max:100',

                'descricao_projecto'  => 'required|string|min:20',
                'imagem_projecto'     => 'required|image|mimes:jpeg,png,jpg,webp|max:2048',
                'galeria'     => 'nullable',

                'data_projecto'       => 'required|date',
            ],
            [
                /* Nome do projeto */
                'nome_projecto.required' => 'O nome do projecto é obrigatório.',
                'nome_projecto.string'   => 'O nome do projecto deve ser um texto válido.',
                'nome_projecto.max'      => 'O nome do projecto não pode ultrapassar 150 caracteres.',


                /* Endereço */
                'endereco.required' => 'O endereço do projecto é obrigatório.',
                'endereco.string'   => 'O endereço deve ser um texto válido.',
                'endereco.max'      => 'O endereço não pode ultrapassar 255 caracteres.',

                /* Cliente */
                'nome_cliente.required' => 'O nome do cliente é obrigatório.',
                'nome_cliente.string'   => 'O nome do cliente deve ser um texto válido.',
                'nome_cliente.max'      => 'O nome do cliente não pode ultrapassar 150 caracteres.',

                /* Engenheiro responsável */
                'engenheiro_resp.required' => 'O engenheiro responsável é obrigatório.',
                'engenheiro_resp.string'   => 'O nome do engenheiro responsável deve ser um texto válido.',
                'engenheiro_resp.max'      => 'O nome do engenheiro responsável não pode ultrapassar 150 caracteres.',

                /* Orçamento */
                'orcamento_prev.required' => 'O orçamento previsto é obrigatório.',
                'orcamento_prev.numeric'  => 'O orçamento previsto deve ser um valor numérico.',
                'orcamento_prev.min'      => 'O orçamento previsto não pode ser negativo.',

                /* Custo total */
                'custo_total.numeric' => 'O custo total deve ser um valor numérico.',
                'custo_total.min'     => 'O custo total não pode ser negativo.',

                /* Moeda */
                'moeda.required' => 'A moeda é obrigatória.',
                'moeda.string'   => 'A moeda deve ser um texto válido.',
                'moeda.max'      => 'A moeda não pode ultrapassar 10 caracteres.',

                /* Estado */
                'estado.required' => 'O estado do projecto é obrigatório.',
                'estado.string'   => 'O estado do projecto deve ser um texto válido.',
                'estado.max'      => 'O estado do projecto não pode ultrapassar 50 caracteres.',

                /* Tipo de serviço */
                'tipo_servico.required' => 'O tipo de serviço é obrigatório.',
                'tipo_servico.string'   => 'O tipo de serviço deve ser um texto válido.',
                'tipo_servico.max'      => 'O tipo de serviço não pode ultrapassar 100 caracteres.',

                /* Tipo de projecto */
                'tipo_projecto.required' => 'O tipo de projecto é obrigatório.',
                'tipo_projecto.string'   => 'O tipo de projecto deve ser um texto válido.',
                'tipo_projecto.max'      => 'O tipo de projecto não pode ultrapassar 100 caracteres.',

                /* Descrição */
                'descricao_projecto.required' => 'A descrição do projecto é obrigatória.',
                'descricao_projecto.string'   => 'A descrição do projecto deve ser um texto válido.',
                'descricao_projecto.min'      => 'A descrição do projecto deve ter no mínimo 20 caracteres.',

                /* Imagem */
                'imagem_projecto.image' => 'O ficheiro enviado deve ser uma imagem.',
                'imagem_projecto.mimes' => 'A imagem deve estar nos formatos: jpeg, png, jpg ou webp.',
                'imagem_projecto.max'   => 'A imagem não pode ter mais que 2MB.',

                /* Data */
                'data_projecto.required' => 'A data do projecto é obrigatória.',
                'data_projecto.date'     => 'A data do projecto deve ser uma data válida.',
            ]
        );

        if ($validator->fails()) {
            return redirect()->back()
                ->withInput()
                ->with([
                    'toast_message' => $validator->errors()->first(),
                    'toast_type' => 'error'
                ]);

        }

        $dados = $request->only([
            'nome_projecto', 'data_projecto',
            'custo_total', 'orcamento_prev',
            'tipo_servico', 'tipo_projecto',
            'descricao_projecto', 'endereco',
            'moeda', 'estado',
            'nome_cliente', 'engenheiro_resp'
        ]);

        $tiposParaSiglas = [
            'design_interior' => 'DIN',
            'obras_publicas'  => 'OPUB',
            'reformas'        => 'REF',
            'arquitetura'     => 'DARQ',
            'projecto_engenhraira'  => 'PENG',

        ];


        $sigla = $tiposParaSiglas[$dados['tipo_servico']] ?? null;

        $contaProjecto = Projecto::count() + 1;
        $numeroFormatado = str_pad($contaProjecto, 4, '0', STR_PAD_LEFT);

        $dados['orcamento_prev'] = str_replace(['.', ','], ['', '.'], $dados['orcamento_prev']);
        $dados['custo_total'] = str_replace(['.', ','], ['', '.'], $dados['custo_total']);

        if ($request->hasFile('imagem_projecto')) {

            $path = $request->file('imagem_projecto')->store('projectos', 'public');

            $dados['imagem_projecto'] = $path;
        }
        DB::transaction(function() use($dados, $request, $numeroFormatado, $sigla){
            $projecto = Projecto::create([
            'nome_projecto' => $dados['nome_projecto'],
            'referencia_projecto' => $numeroFormatado . $sigla,
            'endereco' => $dados['endereco'],
            'nome_cliente' => $dados['nome_cliente'],
            'engenheiro_resp' => $dados['engenheiro_resp'],
            'orcamento_prev' => $dados['orcamento_prev'],
            'custo_total' => $dados['custo_total'],
            'moeda' => $dados['moeda'],
            'estado' => $dados['estado'],
            'tipo_servico' => $dados['tipo_servico'],
            'tipo_projecto' => $dados['tipo_projecto'],
            'descricao_projecto' => $dados['descricao_projecto'],
            'imagem_projecto' => $dados['imagem_projecto'],
            'data_projecto' => $dados['data_projecto']
        ]);

        if ($request->hasFile('galeria')) {
            foreach($request->file('galeria') as $file  ){
                if (!$file->isValid()) continue;

                $path = $file->store('projectos', 'public');

                $projecto->galeria()->create([
                    'path_imagem' => $path,
                    'tipo_galeria' => 'capa'
                ]);
            }
        }
        });

        return redirect()->route('projectos.index')->with([
            'toast_message'=> 'Projecto adicionado com sucesso!',
            'toast_type'=> 'success'
        ]);

    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
