<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Obra;
use App\Models\Cliente;
use Illuminate\Support\Facades\Validator;

class ObrasController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $search = $request->input('search');

        $obras = Obra::query()->when($search, function ($query, $search) {
            $query->Where('nome_obra', '=', "%{$search}%")
            ->orWhere('tipo_servico', '=', "%{$search}%")
            ->orWhere('ref_obra', '=', "%{$search}%")
            ->orWhere('tipo_projecto', '=', "%{$search}%");
        })->paginate(10);
        return view('dashboard.admin.pages.obras.obras', ['obras' => $obras]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $clientes = Cliente::with('usuario')->get();
        return view('dashboard.admin.pages.obras.add-obras', ['clientes' => $clientes]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [
                'nome_obra'       => 'required|string|max:50',
                'ref_obra'            => 'required|string|max:200',
                'cliente_id'        => 'nullable|int',
                'orcamento'      => 'required|numeric|min:0',
                'custo_total'         => 'required|numeric|min:0',

                'moeda'               => 'required|string|max:20',

                'tipo_servico'        => 'required|string|max:100',
                'tipo_projecto'       => 'required|string|max:100',

                'data_inicio'       => 'required|date',
            ],
            [
                /* Nome da obra */
                'nome_obra.required' => 'O nome da obra é obrigatório.',
                'nome_obra.string'   => 'O nome da obra deve ser um texto válido.',
                'nome_obra.max'      => 'O nome da obra não pode ultrapassar 150 caracteres.',

                /* Nome da obra */
                'ref_obra.required' => 'A referência da obra é obrigatório.',
                'ref_obra.string'   => 'A referência da obra deve ser um texto válido.',
                'ref_obra.max'      => 'A referência da obra não pode ultrapassar 150 caracteres.',



                /* Orçamento */
                'orcamento.required' => 'O orçamento previsto é obrigatório.',
                'orcamento.numeric'  => 'O orçamento previsto deve ser um valor numérico.',
                'orcamento.min'      => 'O orçamento previsto não pode ser negativo.',

                /* Custo total */
                'custo_total.numeric' => 'O custo total deve ser um valor numérico.',
                'custo_total.min'     => 'O custo total não pode ser negativo.',

                /* Moeda */
                'moeda.required' => 'A moeda é obrigatória.',
                'moeda.string'   => 'A moeda deve ser um texto válido.',
                'moeda.max'      => 'A moeda não pode ultrapassar 10 caracteres.',



                /* Tipo de serviço */
                'tipo_servico.required' => 'O tipo de serviço é obrigatório.',
                'tipo_servico.string'   => 'O tipo de serviço deve ser um texto válido.',
                'tipo_servico.max'      => 'O tipo de serviço não pode ultrapassar 100 caracteres.',

                /* Tipo de projecto */
                'tipo_projecto.required' => 'O tipo de projecto é obrigatório.',
                'tipo_projecto.string'   => 'O tipo de projecto deve ser um texto válido.',
                'tipo_projecto.max'      => 'O tipo de projecto não pode ultrapassar 100 caracteres.',



                /* Data */
                'data_inicio.required' => 'A data de inicio é obrigatória.',
                'data_inicio.date'     => 'A data de inicio deve ser uma data válida.',
            ]
        );

        if ($validator->fails()) {
            return redirect()->back()
                ->withInput()
                ->with([
                    'toast_message' => $validator->errors()->first(),
                    'toast_type' => 'error'
                ]);

        }

        $dados = $request->only([
            'nome_obra', 'ref_obra',
            'data_inicio', 'custo_total',
            'orcamento', 'tipo_servico',
            'tipo_projecto', 'moeda',
            'cliente_id'
        ]);

        Obra::create([
            'nome_obra' => $dados['nome_obra'],
            'ref_obra'=>$dados['ref_obra'],
            'orcamento' => $dados['orcamento'],
            'custo_total' => $dados['custo_total'],
            'moeda' => $dados['moeda'],
            'tipo_servico' => $dados['tipo_servico'],
            'tipo_projecto' => $dados['tipo_projecto'],
            'data_inicio' => $dados['data_inicio'],
            'cliente_id' => $dados['cliente_id'] ?? null
        ]);

        return redirect()->route('obras.index')->with([
            'toast_message'=> 'Obra adicionada com sucesso!',
            'toast_type'=> 'success'
        ]);
    }
    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $obra = Obra::find($id);

       if($obra->cliente_id == null){
         $clientes = Cliente::with('usuario')->get();
       }
        return view('dashboard.admin.pages.obras.add-obras', ['clientes' => $clientes, 'obra' => $obra]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }

    public function buscarClientes(Request $request)
    {
        $clientes = Cliente::with('usuario')
            ->when($request->search, function ($q) use ($request) {
                $q->where('nome_cliente', 'like', '%' . $request->search . '%');
            })
            ->get();

        return view('dashboard.admin.pages.obras._includes.lista-cliente', compact('clientes'));
    }
}
