<?php

namespace App\Http\Controllers;

use App\Mail\SenhaAcessoClienteMail;
use App\Models\Candidatura;
use App\Models\Cliente;
use App\Models\Usuario;
use App\Models\Vaga;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;

class CandidaturasController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $search = $request->input('search');
     
        $candidaturas = Candidatura::query()->when($search, function ($query, $search) {
            $query->Where('nome_candidato', 'LIKE', "%{$search}%")
                    ->orWhere('email_candidato', 'LIKE',"%{$search}%" )
                    ->orWhere('bilhete_candidato', 'LIKE',"%{$search}%" )
                    ->orWhere('numero_registro', 'LIKE',"%{$search}%" );
        })->paginate(10);
    
        return view('dashboard.admin.pages.candidaturas.candidaturas', ['candidaturas' => $candidaturas]);
    }


    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nome_candidato' => 'required',
            'email_candidato' => 'required|email',
            'bilhete_candidato' => 'required',
            'telefone_candidato'=> 'required',
            'vaga_id' => 'required',
            ],
            [
                'nome_candidato.required' => 'O nome  é obrigatório.',
                
                'email_candidato.required' => 'O email é obrigatório.',
                
                'bilhete_candidato.required'=>'O número do bilhete é obrigatório',
                
                'telefone_candidato.required' => 'O telefone é obrigatório',

                'vaga_id.required'=>'Tem que selecionar uma vaga',
                
            ]
        );

        if ($validator->fails()) {
            return redirect()->back()
                ->withInput()
                ->with([
                    'toast_message' => $validator->errors()->first(),
                    'toast_type' => 'error'
                ]);
        
        }

        $numeroAleatorio = str_pad(mt_rand(0, 9999999999), 10, '0', STR_PAD_LEFT);
        $total_candidatura = Candidatura::count() + 1;
        $numero_registro = $numeroAleatorio. '-' . str_pad($total_candidatura, 2, '0', STR_PAD_LEFT);
        $senhaTemp = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
        $candidatura = null;

        DB::transaction(function() use ($request, $numero_registro, $senhaTemp, $candidatura ){
            if($request->boolean('criar_conta')){
                
                $usuario = Usuario::create([
                    'email_acesso' => $request->email_candidato,
                    'password'=> $senhaTemp,
                    'status' => true,
                    'painel' => 'Cliente'
                ]);

                $cliente = $usuario->cliente()->create([
                    'nome_cliente' => $request->nome_candidato,
                    'email_cliente' => $request->email_candidato,
                    'bilhete_cliente'=> $request->bilhete_candidato,
                ]);

                $candidatura = $cliente->candidatura()->create([
                    'nome_candidato' => $request->nome_candidato,
                    'email_candidato' => $request->email_candidato,
                    'bilhete_candidato'=> $request->bilhete_candidato,
                    'telefone_candidato'=> $request->telefone_candidato,
                    'numero_registro' => $numero_registro,
                    'vaga_id' => $request->vaga_id,
                    'descricao_candidatura'=> $request->descricao_candidatura 
                ]);

                $this->enviar_email($request->email_candidato, $senhaTemp);
            }else{
                $candidatura = Candidatura::create([
                    'nome_candidato' => $request->nome_candidato,
                    'email_candidato' => $request->email_candidato,
                    'bilhete_candidato'=> $request->bilhete_candidato,
                    'telefone_candidato'=> $request->telefone_candidato,
                    'numero_registro' => $numero_registro,
                    'vaga_id' => $request->vaga_id, 
                ]);
            }

            if ($request->hasFile('doc_anexo')){
                foreach ($request->file('doc_anexo') as $item) {
                    $path = $item->store('anexos', 'public');
                    $candidatura->anexo()->create([
                        'doc_anexo'=>$path
                    ]);
                }
            }

        });
        
        
        return redirect()->back()->with([
             'toast_message' => 'Candidatura enviada',
             'toast_type' => 'success'
        ]);
         
    }

    public function update(Request $request, string $id){
        $candidatura = Candidatura::find($id);

        if($candidatura){
            $candidatura->update([
                'status' => $request->status
            ]);

            return redirect()->back()->with([
                'toast_message' => 'Candidatura actualizada',
                'toast_type' => 'success'
            ]);
        }
    }
    /**
     *
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $candidatura = Candidatura::find($id);

        if($candidatura){
            $candidatura->delete();

            return redirect()->back()->with([
                'toast_message' => 'Candidatura eliminada',
                'toast_type' => 'success'
            ]);
        }
    }

    public function enviar_email(string $email, string $senha){
        if($email){
            // $email_criptografado = Crypt::encryptString($email);
            try{
                Mail::to($email)->send(new SenhaAcessoClienteMail($email, $senha)); 
            }catch(\Exception $e){
                Log::error('Falha ao enviar senha temporaria: ' . $e->getMessage()); 
            }
        }
    }
}
